const { computerVisionClient } = require("../config/cvHelper");
const UserModel = require("../models/user");
const AltModel = require("../models/altTextSchema");
const { FEATURES } = require("../constants/app.constants");
const { translate } = require("free-translate");
const axios = require("axios");
const sharp = require("sharp");
const fs = require("fs");
const { translatorTextClient } = require("../config/translateHelper");
const user = require("../models/user");

const convertImageType = async (url) => {
  const dir = "./public";

  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir);
  }

  let imageName = "";
  await axios
    .get(url, {
      responseType: "arraybuffer",
    })
    .then(async ({ data }) => {
      /*convert to png */
      const img = await sharp(data).toFormat("png").toBuffer();

      /* Rename and store in server */
      const newImage = Date.now() + ".png";
      try {
        fs.writeFileSync(`${dir}/${newImage}`, img);
        imageName = newImage;
      } catch (e) {
        console.log("Error in writing file ", e.message);
      }
    })
    .catch((e) => console.log(e));

  return imageName;
};

const deleteImage = async (imageName) => {
  /* Delete from server */
  fs.unlink(`./public/${imageName}`, function (err) {
    if (err && err.code == "ENOENT") {
      // file doesn't exist
      console.log("File doesn't exist.");
      return false;
    } else if (err) {
      // other errors, e.g. maybe we don't have enough permission
      console.log("Error occurred while trying to remove file");
      return false;
    } else {
      console.log("Removed");
      return true;
    }
  });
};

const imageAnalysis = async (images) => {
  let altArrayHebrew = [];
  let altArrayError = [];

  return new Promise(async (resolve, reject) => {
    for (let i = 0; i < images.length; i++) {
      const imageURL = images[i];

      const imageName = await convertImageType(imageURL);
      if (imageName) {
        let image = fs.readFileSync(`./public/${imageName}`);

        // For Image Captioning
        await computerVisionClient
          .analyzeImageInStream(image, { visualFeatures: ["Description"] })
          .then(async (res) => {
            console.log(
              `Description: ${res["description"]["captions"][0]["text"]}`
            );
            await translateAltText(res["description"]["captions"][0]["text"])
              .then((translatedText) => {
                altArrayHebrew.push({ src: imageURL, alt: translatedText });
              })
              .catch((e) => {
                altArrayError.push({ src: imageURL, error: e });
              });
            await deleteImage(imageName);
          })
          .catch(async (e) => {
            console.log(e.message);
            altArrayError.push({ src: imageURL, error: e.message });
            await deleteImage(imageName);
          });
      } else {
        altArrayError.push({ src: imageURL, error: "Unable to convert image" });
      }
    }
    resolve(altArrayHebrew);
  });
};

const translateAltText = async (altText) => {
  return new Promise(async (resolve, reject) => {
    await translatorTextClient.translator
      .translate(
        ["he"],
        [
          {
            text: altText,
          },
        ]
      )
      .then((result) => {
        console.log("Translation: ", result[0]?.translations[0]?.text);
        resolve(result[0]?.translations[0]?.text);
      })
      .catch((err) => {
        console.log("An error occurred in translator: ", err.message);
        reject("An error occurred in translator: ", err.message);
      });
  });
};

const processImages = async (userId, siteKey, images) => {
  await AltModel.findOne({ _userId: userId, _siteId: siteKey }).exec(
    async (err, data) => {
      if (err) {
        // res.status(400).json({error: `Error: ${err.message}`})
        console.log(err.message);
      } else if (data) {
        if (data.altTexts && data.altTexts.length > 0) {
          let filteredImages = [];
          images.map((image) => {
            if (!data.altTexts.find(({ src }) => src === image)) {
              filteredImages.push(image);
            }
          });
          const imageAnalysisResponse =
            filteredImages.length > 0
              ? await imageAnalysis(filteredImages)
              : [];
          data.altTexts = data.altTexts.concat(imageAnalysisResponse);
        } else {
          const imageAnalysisResponse = await imageAnalysis(images);
          data.altTexts = imageAnalysisResponse;
        }
        data.save((err, data) => {
          if (err) {
            console.log("Error in saving alt texts: ", err.message);
            // res.status(400).json({error: `Error in saving alt texts ${err.message}`});
          } else if (data) {
            // res.status(200).json({message: `Alt texts updated successfully`});
            console.log("Alt texts updated successfully");
          }
        });
      } else {
        const imageAnalysisResponse = await imageAnalysis(images);
        let newAltObj = new AltModel({
          _userId: userId,
          _siteId: siteKey,
          altTexts: imageAnalysisResponse,
        });
        newAltObj.save((err, data) => {
          if (err) {
            console.log("Error in saving alt texts: ", err.message);
            // res.status(400).json({error: `Error in saving alt texts ${err.message}`})
          } else if (data) {
            // res.status(200).json({message: `Alt texts saved successfully`})
            console.log("Alt texts saved successfully");
          }
        });
      }
    }
  );
};

exports.addAltTexts = async (req, res) => {
  try {
    const { userId, siteKey, images } = req.body;
    UserModel.findOne({ _id: userId }, async (err, user) => {
      if (err) {
        console.log("Error ", err);
        return res.status(400).json({ error: `Error: ${err.message}` });
      } else if (!user) {
        return res.status(400).json({
          error: "Email not Registered. Please Sign Up!",
        });
      } else {
        let result = user?.Sites?.find((site) => site._id == siteKey);
        if (result) {
          if (
            !result.Feature.includes("ALT_TEXT") &&
            !result.Feature.includes("ALL_FEATURES")
          ) {
            return res.status(400).json({
              error: "Sorry! can't find this feature on your site.",
            });
          }
          // Processing the images in background
          processImages(userId, siteKey, images);
          return res
            .status(200)
            .json({ message: "Images received successfully" });
        } else {
          return res
            .status(400)
            .json({
              error: `User with this site key ${siteKey} does not exist.`,
            });
        }
      }
    });
  } catch (e) {
    res.status(400).json({ error: `Error: ${e.message}` });
  }
};

exports.getAltTexts = async (req, res) => {
  try {
    const { userId, siteKey } = req.body;
    UserModel.findOne({ _id: userId }, async (err, user) => {
      if (err) {
        console.log("Error ", err);
        return res.status(400).json({ error: `Error: ${err.message}` });
      } else if (!user) {
        return res.status(400).json({
          error: "Email not Registered. Please Sign Up!",
        });
      } else {
        let result = user?.Sites?.find((site) => site._id == siteKey);
        if (result) {
          if (
            !result.Feature.includes("ALT_TEXT") &&
            !result.Feature.includes("ALL_FEATURES")
          ) {
            return res.status(400).json({
              error: "Unable to fetch, can't find this feature on your site.",
            });
          }
          await AltModel.findOne({
            _userId: user._id,
            _siteId: result._id,
          }).exec((err, data) => {
            if (err) {
              return res.status(400).json({ error: `Error: ${err.message}` });
            } else if (!data) {
              return res.status(500).json({ error: "No alt texts found" });
            } else {
              return res.status(200).json({ altTexts: data.altTexts });
            }
          });
        } else {
          return res
            .status(400)
            .json({
              error: `User with this site key ${siteKey} does not exit.`,
            });
        }
      }
    });
  } catch (e) {
    res.status(400).json({ error: `Error: ${e.message}` });
  }
};

// exports.getUserImages = async (req, res) => {
//   try {
//     const userId = req?.params?.userId;
//     let userSites=[];
//     UserModel.findOne({_id:userId},'-_id Sites').then((user)=>{
//       return user.Sites.filter((site)=>site.Feature.includes('ALT_TEXT'))
//     }).then((filteredSites)=>{
//       const sitesID=filteredSites.map(site=>site._id);
//       return AltModel.find({'_siteId': { $in: sitesID}})
//     }).then((result)=>{
//       // console.log(sitesID);
//       res.status(200).json({'sites':result})
//     })
//     .catch((error)=>{
//       res.status(400).json({'error':error.message})
//     })
//   } catch (error) {
//     res.status(400).json({ error: error.message });
//   }
// };
